"use client";
import Image from "next/image";
import Link from "next/link";
import { useLocale, useTranslations } from "next-intl";
import { Phone, Mail, MapPin } from "lucide-react";

type Props = {
  logoSrc: string;
  alt?: string;
  phones?: { label: string; href: string }[];
  email?: string;
  addressHtml?: string;
  className?: string;
};

export default function Footer({
  logoSrc,
  alt = "Fast Express Shipping",
  phones = [
    { label: "+971 544 149 643", href: "tel:+971 544 149 643" },
    { label: "+971 568 410 440", href: "tel:+971 568 410 440" },
  ],
  email = "FASTEXPRESS-UAE.COM",
  addressHtml = "United Arab Emirates - Dubai - WHG3 Warehouse, Dry Port",
  className = "",
}: Props) {
  const t = useTranslations("Footer");
  const locale = useLocale();
  const isRTL = locale === "ar";
  const year = new Date().getFullYear();

  return (
    <footer className={`bg-[#2f2f2f] text-white/90 pt-10 pb-6 ${className}`}>
      <div className="custom-container">
        <p
          className={`h4Text tracking-wide text-white/95 mb-8 ${
            isRTL ? "text-center" : "text-center"
          }`}
        >
          {t("tagline")}
        </p>

        <div className="grid grid-cols-1 md:grid-cols-2 gap-10 items-start">
          <div className="flex flex-col items-start gap-4">
            <Image
              src={logoSrc}
              alt={alt}
              width={84}
              height={62}
              className="h-14 w-auto object-contain"
              priority
            />
            <div>
              <h3 className="h4Text text-white">{t("brand")}</h3>
            </div>
          </div>

          <div
            className={`flex flex-col gap-3 ${
              isRTL ? "items-end" : "items-end"
            }`}
          >
            <div className="flex flex-wrap items-center gap-x-3 gap-y-2">
              <Phone className="h-4 w-4 opacity-80" />
              {phones.map((p, i) => (
                <span key={i} className="text-sm md:text-base">
                  <Link
                    href={p.href}
                    dir="ltr"
                    className="hover:underline inline-block"
                  >
                    {p.label}
                  </Link>
                  {i < phones.length - 1 && <span className="mx-2">-</span>}
                </span>
              ))}
            </div>

            <div className="flex items-center gap-3">
              <Mail className="h-4 w-4 opacity-80" />
              <Link
                href={`mailto:${email}`}
                className="text-sm md:text-base hover:underline"
              >
                {email}
              </Link>
            </div>

            <div className="flex items-center gap-3">
              <MapPin className="h-4 w-4 opacity-80" />
              <span
                className="text-sm md:text-base"
                dangerouslySetInnerHTML={{ __html: addressHtml }}
              />
            </div>
          </div>
        </div>

        <hr className="border-white/20 my-6" />

        <div className="flex flex-col md:flex-row items-center justify-between gap-3">
          <p className="text-xs md:text-sm text-white/70">
            © {year} {t("brand")}. {t("rights")}
          </p>
          <p className="text-xs md:text-sm text-white/70">
            {t("powered")} <span className="font-semibold">UnoCode</span>
          </p>
        </div>
      </div>
    </footer>
  );
}
