"use client";
import React from "react";
import Image from "next/image";
import { CheckCircle } from "lucide-react";
import { motion } from "framer-motion";
import { useLocale } from "next-intl";

export type IraqItem = {
  title: string;
  description: string;
  icon?: React.ReactNode;
};

export type IraqSectionProps = {
  title: string;
  items: IraqItem[];
  image: { src: string; alt?: string; width?: number; height?: number };
  flag: { src: string; alt?: string; width?: number; height?: number };
  imageLeft?: boolean;
  className?: string;
  phoneLabel?: string;
  phoneNumber?: string;
};

export default function IraqSection({
  title,
  items,
  image,
  flag,
  imageLeft = true,
  phoneLabel = "",
  phoneNumber = "",
  className = "",
}: IraqSectionProps) {
  const locale = useLocale();
  const isArabic = locale === "ar";

  const orderClasses = imageLeft
    ? "lg:grid-cols-[1.05fr_1fr]"
    : "lg:grid-cols-[1fr_1.05fr] lg:[&>div:first-child]:order-2";

  return (
    <section className={`my-[5rem] md:my-[7rem] ${className}`}>
      <motion.h2
        initial={{ opacity: 0, y: 10 }}
        whileInView={{ opacity: 1, y: 0 }}
        viewport={{ once: true }}
        transition={{ duration: 0.35 }}
        className="text-3xl sm:text-4xl font-extrabold tracking-tight text-secondary mb-4"
      >
        {title}
      </motion.h2>
      <hr className="border-slate-200 mb-2 md:w-[50%]" />

      <div className={`grid grid-cols-1 ${orderClasses} gap-10 items-center`}>
        <div className="flex flex-col text-start">
          <div className="mt-2 space-y-8">
            {items.map((item, idx) => (
              <motion.div
                key={idx}
                initial={{ opacity: 0, y: 8 }}
                whileInView={{ opacity: 1, y: 0 }}
                viewport={{ once: true }}
                transition={{ duration: 0.25, delay: idx * 0.05 }}
              >
                <div className="flex items-center gap-3">
                  <span className="text-teal-700">
                    {item.icon ?? (
                      <CheckCircle className="h-6 w-6" aria-hidden />
                    )}
                  </span>
                  <h3 className="text-lg md:text-xl font-bold tracking-wide text-primary">
                    {item.title}
                  </h3>
                </div>
                <p className="mt-2 text-[15px] sm:text-base leading-relaxed text-slate-600">
                  {item.description}
                </p>
              </motion.div>
            ))}
            {phoneNumber && (
              <div
                className={`p-4 bg-[#F8FAFC] w-full rounded-md flex items-center ${
                  isArabic ? "flex-row text-left" : "flex-row text-left"
                }`}
              >
                <span
                  className={`text-[#F97316] flex-shrink-0 ${
                    isArabic ? "ml-3" : "mr-3"
                  }`}
                >
                  <svg
                    xmlns="http://www.w3.org/2000/svg"
                    fill="none"
                    viewBox="0 0 24 24"
                    strokeWidth={2}
                    stroke="currentColor"
                    className="w-5 h-5"
                  >
                    <path
                      strokeLinecap="round"
                      strokeLinejoin="round"
                      d="M2.25 6.75c0 8.284 6.716 15 15 15h1.5a2.25 2.25 0 002.25-2.25v-1.372a1.125 1.125 0 00-.852-1.09l-4.548-1.137a1.125 1.125 0 00-1.092.285l-.97.97a.75.75 0 01-.906.138 12.035 12.035 0 01-5.237-5.237.75.75 0 01.138-.906l.97-.97a1.125 1.125 0 00.285-1.092L6.962 3.852A1.125 1.125 0 005.872 3H4.5A2.25 2.25 0 002.25 5.25v1.5z"
                    />
                  </svg>
                </span>

                <div
                  className={`flex items-center gap-2 ${
                    isArabic ? "flex-row" : "flex-row"
                  }`}
                >
                  <span className="text-slate-700 text-lg font-medium">
                    {phoneLabel}:
                  </span>

                  <span
                    className="text-slate-700 text-lg font-medium"
                    dir="ltr"
                  >
                    {phoneNumber}
                  </span>
                </div>
              </div>
            )}
          </div>
        </div>

        <div className="relative">
          <motion.div
            initial={{ opacity: 0, y: 12 }}
            whileInView={{ opacity: 1, y: 0 }}
            viewport={{ once: true }}
            transition={{ duration: 0.35 }}
            className="overflow-hidden rounded-lg"
          >
            <Image
              src={image.src}
              alt={image.alt || "Iraq Shipping"}
              width={image.width || 980}
              height={image.height || 640}
              className="w-full h-auto object-cover rounded-lg"
              priority
            />
          </motion.div>

          <div
            className={`absolute -bottom-8 ${
              isArabic ? "left-8" : "right-8"
            } rounded-md overflow-hidden shadow-lg`}
          >
            <Image
              src={flag.src}
              alt={flag.alt || (isArabic ? "علم العراق" : "Iraq Flag")}
              width={flag.width || 180}
              height={flag.height || 112}
              className="w-[160px] sm:w-[180px] h-auto"
            />
          </div>
        </div>
      </div>
    </section>
  );
}
