"use client";
import React from "react";
import Image from "next/image";
import { CheckCircle } from "lucide-react";
import { motion } from "framer-motion";
import { useLocale } from "next-intl";

export type ShippingItem = {
  title: string;
  description: string;
  icon?: React.ReactNode;
};

export type ShippingSectionProps = {
  title: string;
  image: { src: string; alt?: string; width?: number; height?: number };
  badgeImage?: { src: string; alt?: string; width?: number; height?: number };
  items: ShippingItem[];
  imageLeft?: boolean;
  className?: string;
};

export default function ShippingSection({
  title,
  image,
  badgeImage,
  items,
  imageLeft = true,
  className = "",
}: ShippingSectionProps) {
  const locale = useLocale();
  const isArabic = locale === "ar";

  const orderClasses = imageLeft
    ? "lg:grid-cols-[1.2fr_1fr]"
    : "lg:grid-cols-[1fr_1.2fr] lg:[&>div:first-child]:order-2";

  return (
    <section className={`${className} my-[7rem]`}>
      <div
        className={`flex flex-col lg:grid ${orderClasses} gap-8 items-stretch`}
      >
        <div className="relative basis-1/2">
          <motion.div
            initial={{ opacity: 0, y: 20 }}
            whileInView={{ opacity: 1, y: 0 }}
            viewport={{ once: true }}
            transition={{ duration: 0.4 }}
            className="overflow-hidden h-full"
          >
            <Image
              src={image.src}
              alt={image.alt || "Shipping image"}
              width={image.width || 980}
              height={image.height || 640}
              className="h-full w-full object-cover rounded-lg"
              priority
            />
          </motion.div>

          {badgeImage && (
            <div
              className={`absolute md:top-[77%] max-md:hidden ${
                isArabic ? "right-4" : "left-4"
              }`}
            >
              <div className="overflow-hidden rounded-md shadow-lg">
                <Image
                  src={badgeImage.src}
                  alt={badgeImage.alt || "badge"}
                  width={badgeImage.width || 180}
                  height={badgeImage.height || 120}
                  className="h-auto w-[150px] sm:w-[180px]"
                />
              </div>
            </div>
          )}
        </div>

        {/* TEXT */}
        <div
          className={`flex flex-col ${
            isArabic ? "justify-start" : "justify-center"
          } text-start basis-1/2`}
        >
          <motion.h2
            initial={{ opacity: 0, y: 12 }}
            whileInView={{ opacity: 1, y: 0 }}
            viewport={{ once: true }}
            transition={{ duration: 0.35 }}
            className="text-3xl font-extrabold text-secondary sm:text-4xl mb-6"
          >
            {title}
          </motion.h2>
          <hr className="border-slate-200 mb-2" />

          <div className="space-y-6">
            {items.map((item, idx) => (
              <motion.div
                key={idx}
                initial={{ opacity: 0, y: 10 }}
                whileInView={{ opacity: 1, y: 0 }}
                viewport={{ once: true }}
                transition={{ duration: 0.3, delay: idx * 0.05 }}
              >
                <div className="flex items-center gap-3">
                  <span className="text-teal-700">
                    {item.icon ?? (
                      <CheckCircle className="h-6 w-6" aria-hidden />
                    )}
                  </span>
                  <h3 className="text-xl font-bold text-primary">
                    {item.title}
                  </h3>
                </div>
                <p className="mt-2 text-lg leading-7 text-slate-600">
                  {item.description}
                </p>
              </motion.div>
            ))}
          </div>
        </div>
      </div>
    </section>
  );
}
